//--------------------------------------------------------------
// File     : stm32_ub_dcf77.c
// Datum    : 31.12.2015
// Version  : 1.0
// Autor    : UB
// EMail    : mc-4u(@)t-online.de
// Web      : www.mikrocontroller-4u.de
// CPU      : STM32F4
// IDE      : CooCox CoIDE 1.7.8
// GCC      : 4.9 2015q2
// Module   : GPIO, TIM, EXTI, SYSCFG, MISC
// Funktion : auswertung vom DCF-77 Signal an PB0
//
// Hinweis  : Das DCF-77 Signal wird per ExtInterrupt-0
//            ausgewertet, muss also an PA0, PB0, PC0...PI0 liegen
//
//            Timer2 wird zur Zeitmessung benutzt
//--------------------------------------------------------------


//--------------------------------------------------------------
// Includes
//--------------------------------------------------------------
#include "stm32_ub_dcf77.h"



//--------------------------------------------------------------
// interne Funktionen
//--------------------------------------------------------------
void P_init_TIM(void);
void P_init_EXTI(void);
void P_Store_Value(uint8_t value, uint8_t pos);


//--------------------------------------------------------------
// interne Globale Variabeln
//--------------------------------------------------------------
uint32_t dcf77_1ms_value=0;
uint32_t dcf77_ms_counter=0;
DCF77_interal_t dcf77_int;


//--------------------------------------------------------------
// init vom DCF-77 Modul
//--------------------------------------------------------------
void UB_DCF77_Init(void)
{
  // init vom Timer
  P_init_TIM();

  // init vom Exti-Pin
  P_init_EXTI();

  dcf77_1ms_value=0;
  dcf77_ms_counter=0;

  dcf77_int.mode=M77_Init;
  dcf77_int.ok=0;
  dcf77_int.min=0;
  dcf77_int.std=0;
  dcf77_int.tag=0;
  dcf77_int.monat=0;
  dcf77_int.jahr=0;

  DCF77_TIME.sek=0;
  DCF77_TIME.min=0;
  DCF77_TIME.std=0;
  DCF77_TIME.tag=0;
  DCF77_TIME.monat=0;
  DCF77_TIME.jahr=0;

  // Timer enable
  TIM_Cmd(TIM2, ENABLE);
}


//--------------------------------------------------------------
// aktuelle Zeit vom DCF77-Modul auslesen
// ret_wert :
//   DCF77_NO_SIGNAL  = es wird kein Signal empfangen
//   DCF77_READING    = signal wird empfangen, warte auf uhrzeit
//   DCF77_TIME_ERROR = fehler beim empfang
//   DCF77_TIME_OK    = ok, Uhrzeit steht in "DCF77_TIME"
//--------------------------------------------------------------
DCF77_Status_t UB_DCF77_ReadTime(void)
{
  DCF77_Status_t ret_wert=DCF77_NO_SIGNAL;

  if(dcf77_int.ok==1) {
    ret_wert=DCF77_TIME_OK;
  }
  else {
    if((dcf77_int.mode==M77_Init) || (dcf77_int.mode==M77_Wait4Signal)) {
      ret_wert=DCF77_NO_SIGNAL;
    }
    else if(dcf77_int.mode==M77_Error) {
      ret_wert=DCF77_TIME_ERROR;
    }
    else {
      ret_wert=DCF77_READING;
    }
  }

  return(ret_wert);
}

//--------------------------------------------------------------
// interne Funktion zum init vom Timer
//--------------------------------------------------------------
void P_init_TIM(void)
{
  TIM_TimeBaseInitTypeDef  TIM_TimeBaseStructure;
  NVIC_InitTypeDef NVIC_InitStructure;

  // Clock enable
  RCC_APB1PeriphClockCmd(RCC_APB1Periph_TIM2, ENABLE);

  // Timer disable
  TIM_Cmd(TIM2, DISABLE);

  // Timer init (1ms)
  TIM_TimeBaseStructure.TIM_Period =  (1000-1);
  TIM_TimeBaseStructure.TIM_Prescaler = (84-1);
  TIM_TimeBaseStructure.TIM_ClockDivision = TIM_CKD_DIV1;
  TIM_TimeBaseStructure.TIM_CounterMode = TIM_CounterMode_Up;
  TIM_TimeBaseInit(TIM2, &TIM_TimeBaseStructure);

  // Timer preload enable
  TIM_ARRPreloadConfig(TIM2, ENABLE);

  // Update Interrupt enable
  TIM_ITConfig(TIM2,TIM_IT_Update,ENABLE);

  // NVIC konfig
  NVIC_InitStructure.NVIC_IRQChannel = TIM2_IRQn;
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = 0;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0;
  NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
  NVIC_Init(&NVIC_InitStructure);
}

//--------------------------------------------------------------
// interne Funktion zum init vom Ext-Interrupt
//--------------------------------------------------------------
void P_init_EXTI(void)
{
  GPIO_InitTypeDef   GPIO_InitStructure;
  EXTI_InitTypeDef   EXTI_InitStructure;
  NVIC_InitTypeDef   NVIC_InitStructure;

  // Clock enable (GPIO)
  RCC_AHB1PeriphClockCmd(DCF77_CLOCK, ENABLE);

  // Config als Digital-Eingang
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_IN;
  GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_DOWN;
  GPIO_InitStructure.GPIO_Pin = DCF77_PIN;
  GPIO_Init(DCF77_PORT, &GPIO_InitStructure);

  // Clock enable (SYSCONFIG)
  RCC_APB2PeriphClockCmd(RCC_APB2Periph_SYSCFG, ENABLE);

  // EXT_INT0 mit Pin verbinden
  SYSCFG_EXTILineConfig(DCF77_SOURCE, EXTI_PinSource0);

  // EXT_INT0 config
  EXTI_InitStructure.EXTI_Line = EXTI_Line0;
  EXTI_InitStructure.EXTI_Mode = EXTI_Mode_Interrupt;
  EXTI_InitStructure.EXTI_Trigger = EXTI_Trigger_Rising_Falling;
  EXTI_InitStructure.EXTI_LineCmd = ENABLE;
  EXTI_Init(&EXTI_InitStructure);

  // NVIC config
  NVIC_InitStructure.NVIC_IRQChannel = EXTI0_IRQn;
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = 0x01;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0x01;
  NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
  NVIC_Init(&NVIC_InitStructure);
}

//--------------------------------------------------------------
// bearbeite ein bit der uhrzeit
//--------------------------------------------------------------
void P_Store_Value(uint8_t value, uint8_t pos)
{
  static uint8_t bytewert=0;

  switch(pos) {
    // minuten
    case 21 : if(value==1) bytewert+=1;break;
    case 22 : if(value==1) bytewert+=2;break;
    case 23 : if(value==1) bytewert+=4;break;
    case 24 : if(value==1) bytewert+=8;break;
    case 25 : if(value==1) bytewert+=10;break;
    case 26 : if(value==1) bytewert+=20;break;
    case 27 :
      if(value==1) bytewert+=40;
      dcf77_int.min=bytewert;
      bytewert=0;
      break;
    // stunden
    case 29 : if(value==1) bytewert+=1;break;
    case 30 : if(value==1) bytewert+=2;break;
    case 31 : if(value==1) bytewert+=4;break;
    case 32 : if(value==1) bytewert+=8;break;
    case 33 : if(value==1) bytewert+=10;break;
    case 34 :
      if(value==1) bytewert+=20;
      dcf77_int.std=bytewert;
      bytewert=0;
      break;
    // tag
    case 36 : if(value==1) bytewert+=1;break;
    case 37 : if(value==1) bytewert+=2;break;
    case 38 : if(value==1) bytewert+=4;break;
    case 39 : if(value==1) bytewert+=8;break;
    case 40 : if(value==1) bytewert+=10;break;
    case 41 :
      if(value==1) bytewert+=20;
      dcf77_int.tag=bytewert;
      bytewert=0;
      break;
    // monat
    case 45 : if(value==1) bytewert+=1;break;
    case 46 : if(value==1) bytewert+=2;break;
    case 47 : if(value==1) bytewert+=4;break;
    case 48 : if(value==1) bytewert+=8;break;
    case 49 :
      if(value==1) bytewert+=10;
      dcf77_int.monat=bytewert;
      bytewert=0;
      break;
    // jahr
    case 50 : if(value==1) bytewert+=1;break;
    case 51 : if(value==1) bytewert+=2;break;
    case 52 : if(value==1) bytewert+=4;break;
    case 53 : if(value==1) bytewert+=8;break;
    case 54 : if(value==1) bytewert+=10;break;
    case 55 : if(value==1) bytewert+=20;break;
    case 56 : if(value==1) bytewert+=40;break;
    case 57 :
      if(value==1) bytewert+=80;
      dcf77_int.jahr=bytewert;
      bytewert=0;
      break;
    // ende
    case 58 :
      bytewert=0;
      dcf77_ms_counter=0; // syncronisiere sek_counter
      DCF77_TIME.sek=0;
      DCF77_TIME.min=dcf77_int.min;
      DCF77_TIME.std=dcf77_int.std;
      DCF77_TIME.tag=dcf77_int.tag;
      DCF77_TIME.monat=dcf77_int.monat;
      DCF77_TIME.jahr=dcf77_int.jahr;
      dcf77_int.mode=M77_Ready;
      dcf77_int.ok=1; // set flag
      break;
  }
}

//--------------------------------------------------------------
// ISR (Timer-2) [1ms]
//--------------------------------------------------------------
void TIM2_IRQHandler(void)
{

  if (TIM_GetITStatus(TIM2, TIM_IT_Update) != RESET) {
    // wenn Interrupt aufgetreten
    TIM_ClearITPendingBit(TIM2, TIM_IT_Update);

    dcf77_ms_counter++;
    if(dcf77_ms_counter>=1000) {
      dcf77_ms_counter=0;
      if(DCF77_TIME.sek<59) DCF77_TIME.sek++;
    }

    dcf77_1ms_value++;
    if(dcf77_int.mode==M77_Init) {
      dcf77_1ms_value=0;
      dcf77_int.mode=M77_Wait4Signal;
    }
    else if(dcf77_int.mode==M77_Wait4Signal) {
      // nothing to do
    }
    else if(dcf77_int.mode==M77_Wait4Sync) {
      if(dcf77_1ms_value>DCF77_SYNC) {
        dcf77_1ms_value=0;
        dcf77_int.mode=M77_SyncOk;
      }
    }
    else if(dcf77_int.mode==M77_SyncOk) {
      if(dcf77_1ms_value>DCF77_TIMOUT) dcf77_int.mode=M77_Error;
    }
    else if(dcf77_int.mode==M77_Wait4Time) {
      if(dcf77_1ms_value>DCF77_TIMOUT) dcf77_int.mode=M77_Error;
    }
    else if(dcf77_int.mode==M77_Ready) {
      if(dcf77_1ms_value>DCF77_TIMOUT) dcf77_int.mode=M77_Error;
    }
    else if(dcf77_int.mode==M77_Error) {
      dcf77_int.ok=0;
    }
  }
}


//--------------------------------------------------------------
// ISR (Exti-0) [Lo+Hi-Flanke]
//--------------------------------------------------------------
void EXTI0_IRQHandler(void)
{
  uint32_t wert, signal_value=0;
  static uint8_t pos=0;

  if(EXTI_GetITStatus(EXTI_Line0) != RESET)
  {
    // wenn Interrupt aufgetreten
    EXTI_ClearITPendingBit(EXTI_Line0);

    wert=GPIO_ReadInputDataBit(DCF77_PORT, DCF77_PIN);
    if(wert==Bit_RESET) {
      // lo Flanke     
      signal_value=dcf77_1ms_value;
      if(dcf77_int.mode==M77_Wait4Time) {
        if((signal_value>=DCF77_LO_MIN) && (signal_value<=DCF77_LO_MAX)) {
          P_Store_Value(0,pos);
          pos++;
          if(dcf77_int.mode==M77_Ready) dcf77_int.mode=M77_Wait4Sync;
        }
        else if((signal_value>=DCF77_HI_MIN) && (signal_value<=DCF77_HI_MAX)) {
          P_Store_Value(1,pos);
          pos++;
          if(dcf77_int.mode==M77_Ready) dcf77_int.mode=M77_Wait4Sync;
        }
        else dcf77_int.mode=M77_Error;
      }
    }
    else {
      // hi Flanke
      dcf77_1ms_value=0;
      if(dcf77_int.mode==M77_SyncOk) {
        pos=0;
        dcf77_int.mode=M77_Wait4Time;
      }
      if((dcf77_int.mode==M77_Wait4Signal) || (dcf77_int.mode==M77_Error)) dcf77_int.mode=M77_Wait4Sync;
    }
  }
}
