//--------------------------------------------------------------
// File     : z80_opcode.h
//--------------------------------------------------------------

#ifndef Z80_OPCODE_H_
#define Z80_OPCODE_H_

#include "stm32f7xx.h"
#include "z80_ub.h"
#include "gameboy_ub.h"
#include "z80_opcode_table.h"



//------------------------------------------------
// NOP, STOP
//------------------------------------------------

#define NOP {z80.reg.pc++;}

#define STOP {z80.status |= STATUS_STOP_FLAG;}

//------------------------------------------------
// ROTATE
//------------------------------------------------

#define RLA(value) { \
z80_byte = (value << 1) | bit0_carry[z80.reg.f]; \
z80.reg.f = flag_carry_0x80[value]; \
value = z80_byte; \
z80.reg.pc++; \
}

#define RLCA(value) { \
z80.reg.f = flag_carry_0x80[value]; \
z80_byte = (value << 1) | (value>>7); \
value = z80_byte; \
z80.reg.pc++; \
}

#define RRA(value) { \
z80_byte = (value >> 1) | bit7_carry[z80.reg.f]; \
z80.reg.f = flag_carry_0x01[value]; \
value = z80_byte; \
z80.reg.pc++; \
}

#define RRCA(value) { \
z80.reg.f = flag_carry_0x01[value]; \
z80_byte = (value >> 1) | (value<<7); \
value = z80_byte; \
z80.reg.pc++; \
}


//------------------------------------------------
// INC
//------------------------------------------------

#define INC_R(value) {value++; z80.reg.f = zero_flag[value] | half_carry_flag_inc[value] | (z80.reg.f & FLAG_C); z80.reg.pc++;}

#define INC_DR(value) {value++; z80.reg.pc++;}

#define INC_A16(adr) { \
z80_byte=RD_BYTE_MEM(adr); \
z80_byte++; \
WR_BYTE_MEM(adr, z80_byte); \
z80.reg.f = zero_flag[z80_byte] | half_carry_flag_inc[z80_byte] | (z80.reg.f & FLAG_C); \
z80.reg.pc++; \
}


//------------------------------------------------
// DEC
//------------------------------------------------

#define DEC_R(value) {value--; z80.reg.f = zero_flag[value] | FLAG_N | half_carry_flag_dec[value] | (z80.reg.f & FLAG_C); z80.reg.pc++;}

#define DEC_DR(value) {value--; z80.reg.pc++;}

#define DEC_A16(adr) { \
z80_byte=RD_BYTE_MEM(adr); \
z80_byte--; \
WR_BYTE_MEM(adr, z80_byte); \
z80.reg.f = zero_flag[z80_byte] | FLAG_N | half_carry_flag_dec[z80_byte] | (z80.reg.f & FLAG_C); \
z80.reg.pc++; \
}

//------------------------------------------------
// ADD
//------------------------------------------------

#define ADD_SP_N8(adr) { \
var8bit.uint8 = RD_BYTE_MEM(adr); \
z80_long = z80.reg.sp + var8bit.sint8; \
z80_word = (uint16_t)(z80_long); \
if(var8bit.sint8 >= 0) { \
	z80.reg.f = ((((z80.reg.sp & 0xFF)+var8bit.sint8) > 0xFF) ? FLAG_C : 0); \
	z80.reg.f |= ((((z80.reg.sp & 0x0F)+(var8bit.sint8 & 0x0F)) > 0x0F) ? FLAG_H : 0); \
} else { \
	z80.reg.f = (((z80_word & 0xFF) <= (z80.reg.sp & 0xFF)) ? FLAG_C : 0); \
	z80.reg.f |= (((z80_word & 0x0F) <= (z80.reg.sp & 0x0F)) ? FLAG_H : 0); \
} \
z80.reg.sp = z80_word; \
z80.reg.pc+=2; \
}

#define ADD_R(value) { \
z80_word = z80.reg.a+value; \
u8_index = ((z80_word & BIT3_MASK8) >> 1) | ((value & BIT3_MASK8) >> 2) | ((z80.reg.a & BIT3_MASK8) >> 3); \
z80.reg.a = (uint8_t)(z80_word); \
z80.reg.f = zero_flag[z80.reg.a] | half_carry_flag_8bit_add[u8_index] | ((z80_word & 0x100) ? FLAG_C : 0); \
z80.reg.pc++; \
}

#define ADC_R(value) { \
z80_word = z80.reg.a+value+bit0_carry[z80.reg.f]; \
u8_index = ((z80_word & BIT3_MASK8) >> 1) | ((value & BIT3_MASK8) >> 2) | ((z80.reg.a & BIT3_MASK8) >> 3); \
z80.reg.a = (uint8_t)(z80_word); \
z80.reg.f = zero_flag[z80.reg.a] | half_carry_flag_8bit_add[u8_index] | ((z80_word & 0x100) ? FLAG_C : 0); \
z80.reg.pc++; \
}

#define ADD_N8(adr) { \
z80_byte = RD_BYTE_MEM(adr); \
z80_word = z80.reg.a+z80_byte; \
u8_index = ((z80_word & BIT3_MASK8) >> 1) | ((z80_byte & BIT3_MASK8) >> 2) | ((z80.reg.a & BIT3_MASK8) >> 3); \
z80.reg.a = (uint8_t)(z80_word); \
z80.reg.f = zero_flag[z80.reg.a] | half_carry_flag_8bit_add[u8_index] | ((z80_word & 0x100) ? FLAG_C : 0); \
z80.reg.pc+=2; \
}

#define ADC_N8(adr) { \
z80_byte = RD_BYTE_MEM(adr); \
z80_word = z80.reg.a+z80_byte+bit0_carry[z80.reg.f]; \
u8_index = ((z80_word & BIT3_MASK8) >> 1) | ((z80_byte & BIT3_MASK8) >> 2) | ((z80.reg.a & BIT3_MASK8) >> 3); \
z80.reg.a = (uint8_t)(z80_word); \
z80.reg.f = zero_flag[z80.reg.a] | half_carry_flag_8bit_add[u8_index] | ((z80_word & 0x100) ? FLAG_C : 0); \
z80.reg.pc+=2; \
}

#define ADD_DD(value) { \
z80_long = z80.reg.hl+value; \
u8_index = ((z80_long & BIT11_MASK16) >> 9) | ((value & BIT11_MASK16) >> 10) | ((z80.reg.hl & BIT11_MASK16) >> 11); \
z80.reg.f = (z80.reg.f & FLAG_Z) | half_carry_flag_16bit_add[u8_index] | ((z80_long & 0x10000) ? FLAG_C : 0); \
z80.reg.hl = (uint16_t)(z80_long); \
z80.reg.pc++; \
}

#define ADD_N16(adr) { \
z80_byte = RD_BYTE_MEM(adr); \
z80_word = z80.reg.a+z80_byte; \
u8_index = ((z80_word & BIT3_MASK8) >> 1) | ((z80_byte & BIT3_MASK8) >> 2) | ((z80.reg.a & BIT3_MASK8) >> 3); \
z80.reg.a = (uint8_t)(z80_word); \
z80.reg.f = zero_flag[z80.reg.a] | half_carry_flag_8bit_add[u8_index] | ((z80_word & 0x100) ? FLAG_C : 0); \
z80.reg.pc++; \
}

#define ADC_N16(adr) { \
z80_byte = RD_BYTE_MEM(adr); \
z80_word = z80.reg.a+z80_byte+bit0_carry[z80.reg.f]; \
u8_index = ((z80_word & BIT3_MASK8) >> 1) | ((z80_byte & BIT3_MASK8) >> 2) | ((z80.reg.a & BIT3_MASK8) >> 3); \
z80.reg.a = (uint8_t)(z80_word); \
z80.reg.f = zero_flag[z80.reg.a] | half_carry_flag_8bit_add[u8_index] | ((z80_word & 0x100) ? FLAG_C : 0); \
z80.reg.pc++; \
}


//------------------------------------------------
// SUB
//------------------------------------------------

#define SUB_R(value) { \
z80_word = z80.reg.a-value; \
u8_index = ((z80_word & BIT3_MASK8) >> 1) | ((value & BIT3_MASK8) >> 2) | ((z80.reg.a & BIT3_MASK8) >> 3); \
z80.reg.a = (uint8_t)(z80_word); \
z80.reg.f = zero_flag[z80.reg.a] | FLAG_N | half_carry_flag_8bit_sub[u8_index] | ((z80_word & 0x100) ? FLAG_C : 0); \
z80.reg.pc++; \
}

#define SBC_R(value) { \
z80_word = z80.reg.a-value-bit0_carry[z80.reg.f]; \
u8_index = ((z80_word & BIT3_MASK8) >> 1) | ((value & BIT3_MASK8) >> 2) | ((z80.reg.a & BIT3_MASK8) >> 3); \
z80.reg.a = (uint8_t)(z80_word); \
z80.reg.f = zero_flag[z80.reg.a] | FLAG_N | half_carry_flag_8bit_sub[u8_index] | ((z80_word & 0x100) ? FLAG_C : 0); \
z80.reg.pc++; \
}

#define SUB_N8(adr) { \
z80_byte = RD_BYTE_MEM(adr); \
z80_word = z80.reg.a-z80_byte; \
u8_index = ((z80_word & BIT3_MASK8) >> 1) | ((z80_byte & BIT3_MASK8) >> 2) | ((z80.reg.a & BIT3_MASK8) >> 3); \
z80.reg.a = (uint8_t)(z80_word); \
z80.reg.f = zero_flag[z80.reg.a] | FLAG_N | half_carry_flag_8bit_sub[u8_index] | ((z80_word & 0x100) ? FLAG_C : 0); \
z80.reg.pc+=2; \
}

#define SUB_N16(adr) { \
z80_byte = RD_BYTE_MEM(adr); \
z80_word = z80.reg.a-z80_byte; \
u8_index = ((z80_word & BIT3_MASK8) >> 1) | ((z80_byte & BIT3_MASK8) >> 2) | ((z80.reg.a & BIT3_MASK8) >> 3); \
z80.reg.a = (uint8_t)(z80_word); \
z80.reg.f = zero_flag[z80.reg.a] | FLAG_N | half_carry_flag_8bit_sub[u8_index] | ((z80_word & 0x100) ? FLAG_C : 0); \
z80.reg.pc++; \
}


#define SBC_N8(adr) { \
z80_byte = RD_WORD_MEM(adr); \
z80_word = z80.reg.a-z80_byte-bit0_carry[z80.reg.f]; \
u8_index = ((z80_word & BIT3_MASK8) >> 1) | ((z80_byte & BIT3_MASK8) >> 2) | ((z80.reg.a & BIT3_MASK8) >> 3); \
z80.reg.a = (uint8_t)(z80_word); \
z80.reg.f = zero_flag[z80.reg.a] | FLAG_N | half_carry_flag_8bit_sub[u8_index] | ((z80_word & 0x100) ? FLAG_C : 0); \
z80.reg.pc+=2; \
}

#define SBC_N16(adr) { \
z80_byte = RD_BYTE_MEM(adr); \
z80_word = z80.reg.a-z80_byte-bit0_carry[z80.reg.f]; \
u8_index = ((z80_word & BIT3_MASK8) >> 1) | ((z80_byte & BIT3_MASK8) >> 2) | ((z80.reg.a & BIT3_MASK8) >> 3); \
z80.reg.a = (uint8_t)(z80_word); \
z80.reg.f = zero_flag[z80.reg.a] | FLAG_N | half_carry_flag_8bit_sub[u8_index] | ((z80_word & 0x100) ? FLAG_C : 0); \
z80.reg.pc++; \
}

//------------------------------------------------
// XOR
//------------------------------------------------


#define XOR_R(value) {z80.reg.a ^= value; z80.reg.f = zero_flag[z80.reg.a]; z80.reg.pc++;}

#define XOR_M(adr) {z80.reg.a ^= RD_BYTE_MEM(adr); z80.reg.f = zero_flag[z80.reg.a]; z80.reg.pc++;}

#define XOR_N8(adr) {z80.reg.a ^= RD_BYTE_MEM(adr); z80.reg.f = zero_flag[z80.reg.a]; z80.reg.pc+=2;}

//------------------------------------------------
// LD
//------------------------------------------------


#define LD_R(dest,source) {dest=source; z80.reg.pc++;}

#define LD_N8(dest,adr) {dest=RD_BYTE_MEM(adr); z80.reg.pc+=2;}

#define LD_HL_R(value) {WR_BYTE_MEM(z80.reg.hl, value); z80.reg.pc++;}

#define LD_R_HL(value) {value=RD_BYTE_MEM(z80.reg.hl); z80.reg.pc++;}

#define LD_M(adr) {z80.reg.a=RD_BYTE_MEM(adr); z80.reg.pc++;}

#define LD_DD_N16(dest,adr) {dest=RD_WORD_MEM(adr); z80.reg.pc+=3;}

#define LD_A16(adr) {WR_BYTE_MEM(adr, z80.reg.a); z80.reg.pc++;}

#define LD_N16_SP(adr) { z80_word = RD_WORD_MEM(adr); WR_WORD_MEM(z80_word, z80.reg.sp); z80.reg.pc+=3;}

#define LD_HL_N8(adr) {z80_byte=RD_BYTE_MEM(adr); WR_BYTE_MEM(z80.reg.hl, z80_byte); z80.reg.pc+=2;}

#define LD_N16_A(adr) {z80_word = RD_WORD_MEM(adr); WR_BYTE_MEM(z80_word, z80.reg.a); z80.reg.pc+=3;}

#define LD_A_N16(adr) {z80_word = RD_WORD_MEM(adr); z80.reg.a = RD_BYTE_MEM(z80_word); z80.reg.pc+=3;}

#define LDH_N8_A(adr) { \
z80_byte = RD_BYTE_MEM(adr); \
WR_BYTE_MEM((0xFF00+z80_byte), z80.reg.a); \
z80.reg.pc+=2; \
}

#define LDH_C { \
WR_BYTE_MEM((0xFF00+z80.reg.c), z80.reg.a); \
z80.reg.pc++; \
}

#define LDH_A_N8(adr) { z80_byte = RD_BYTE_MEM(adr); z80.reg.a = RD_BYTE_MEM((0xFF00+z80_byte)); z80.reg.pc+=2;}

#define LDH_A {z80.reg.a=RD_BYTE_MEM((0xFF00+z80.reg.c)); z80.reg.pc++;}

#define LD_HL_SP_N8(adr) { \
var8bit.uint8 = RD_BYTE_MEM(adr); \
z80_long = z80.reg.sp + var8bit.sint8; \
z80.reg.hl = (uint16_t)(z80_long); \
if(var8bit.sint8 >= 0) { \
	z80.reg.f = ((((z80.reg.sp & 0xFF)+var8bit.sint8) > 0xFF) ? FLAG_C : 0); \
	z80.reg.f |= ((((z80.reg.sp & 0x0F)+(var8bit.sint8 & 0x0F)) > 0x0F) ? FLAG_H : 0); \
} else { \
	z80.reg.f = (((z80.reg.hl & 0xFF) <= (z80.reg.sp & 0xFF)) ? FLAG_C : 0); \
	z80.reg.f |= (((z80.reg.hl & 0x0F) <= (z80.reg.sp & 0x0F)) ? FLAG_H : 0); \
} \
z80.reg.pc+=2; \
}

//------------------------------------------------
// OR
//------------------------------------------------

#define OR_R(value) {z80.reg.a |= value; z80.reg.f = zero_flag[z80.reg.a]; z80.reg.pc++;}

#define OR_M(adr) {z80.reg.a |= RD_BYTE_MEM(adr); z80.reg.f = zero_flag[z80.reg.a]; z80.reg.pc++;}

#define OR_N8(adr) {z80.reg.a |= RD_BYTE_MEM(adr); z80.reg.f = zero_flag[z80.reg.a]; z80.reg.pc+=2;}



//------------------------------------------------
// AND
//------------------------------------------------

#define AND_R(value) {z80.reg.a &= value; z80.reg.f = zero_flag[z80.reg.a] | FLAG_H; z80.reg.pc++;}

#define AND_M(adr) {z80.reg.a &= RD_BYTE_MEM(adr); z80.reg.f = zero_flag[z80.reg.a] | FLAG_H; z80.reg.pc++;}

#define AND_N8(adr) {z80.reg.a &= RD_BYTE_MEM(adr); z80.reg.f = zero_flag[z80.reg.a] | FLAG_H; z80.reg.pc+=2;}


//------------------------------------------------
// CP
//------------------------------------------------


#define CP_R(value) { \
z80_word = z80.reg.a - value; \
z80_byte = (uint8_t)(z80_word); \
u8_index = ((z80_word & BIT3_MASK8) >> 1) | ((value & BIT3_MASK8) >> 2) | ((z80.reg.a & BIT3_MASK8) >> 3); \
z80.reg.f = zero_flag[z80_byte] | FLAG_N | half_carry_flag_8bit_sub[u8_index] | ((z80_word & 0x100) ? FLAG_C : 0); \
z80.reg.pc++; \
}

#define CP_A16(adr) { \
u8_temp = RD_BYTE_MEM(adr); \
z80_word = z80.reg.a - u8_temp; \
z80_byte = (uint8_t)(z80_word); \
u8_index = ((z80_word & BIT3_MASK8) >> 1) | ((u8_temp & BIT3_MASK8) >> 2) | ((z80.reg.a & BIT3_MASK8) >> 3); \
z80.reg.f = zero_flag[z80_byte] | FLAG_N | half_carry_flag_8bit_sub[u8_index] | ((z80_word & 0x100) ? FLAG_C : 0); \
z80.reg.pc++; \
}

#define CPL { \
z80.reg.a = (z80.reg.a ^ 0xFF); \
z80.reg.f |= FLAG_N; \
z80.reg.f |= FLAG_H; \
z80.reg.pc++; \
}

#define CP_N8(adr) { \
u8_temp = RD_BYTE_MEM(adr); \
z80_word = z80.reg.a - u8_temp; \
z80_byte = (uint8_t)(z80_word); \
u8_index = ((z80_word & BIT3_MASK8) >> 1) | ((u8_temp & BIT3_MASK8) >> 2) | ((z80.reg.a & BIT3_MASK8) >> 3); \
z80.reg.f = zero_flag[z80_byte] | FLAG_N | half_carry_flag_8bit_sub[u8_index] | ((z80_word & 0x100) ? FLAG_C : 0); \
z80.reg.pc+=2; \
}



//------------------------------------------------
// JP
//------------------------------------------------


#define JP_HL {z80.reg.pc = z80.reg.hl;}

#define JP_NN(adr) {z80.reg.pc = RD_WORD_MEM(adr);}

#define JP_Z(adr) { \
if((z80.reg.f & FLAG_Z) == FLAG_Z) { \
	z80.reg.pc=RD_WORD_MEM(adr); \
} else { \
	z80.reg.pc+=3; \
} \
}

#define JP_NZ(adr) { \
if((z80.reg.f & FLAG_Z) == 0) { \
	z80.reg.pc=RD_WORD_MEM(adr); \
} else { \
	z80.reg.pc+=3; \
} \
}

#define JP_C(adr) { \
if((z80.reg.f & FLAG_C) == FLAG_C) { \
	z80.reg.pc=RD_WORD_MEM(adr); \
} else { \
	z80.reg.pc+=3; \
} \
}

#define JP_NC(adr) { \
if((z80.reg.f & FLAG_C) == 0) { \
	z80.reg.pc=RD_WORD_MEM(adr); \
} else { \
	z80.reg.pc+=3; \
} \
}

#define JR_N8(adr) {var8bit.uint8 = RD_BYTE_MEM(adr); z80.reg.pc+=var8bit.sint8+2;}

#define JR_NZ(adr) { \
if((z80.reg.f & FLAG_Z) == 0) { \
	var8bit.uint8 = RD_BYTE_MEM(adr); \
	z80.reg.pc+=var8bit.sint8+2; \
} else { \
	z80.reg.pc+=2; \
} \
}

#define JR_Z(adr) { \
if((z80.reg.f & FLAG_Z) == FLAG_Z) { \
	var8bit.uint8 = RD_BYTE_MEM(adr); \
	z80.reg.pc+=var8bit.sint8+2; \
} else { \
	z80.reg.pc+=2; \
} \
}

#define JR_NC(adr) { \
if((z80.reg.f & FLAG_C) == 0) { \
	var8bit.uint8 = RD_BYTE_MEM(adr); \
	z80.reg.pc+=var8bit.sint8+2; \
} else { \
	z80.reg.pc+=2; \
} \
}

#define JR_C(adr) { \
if((z80.reg.f & FLAG_C) == FLAG_C) { \
	var8bit.uint8 = RD_BYTE_MEM(adr); \
	z80.reg.pc+=var8bit.sint8+2; \
} else { \
	z80.reg.pc+=2; \
} \
}

//------------------------------------------------
// CALL
//------------------------------------------------

#define CALL_N16(adr) { \
z80_word = RD_WORD_MEM(adr); \
z80.reg.sp-=2; \
WR_WORD_MEM(z80.reg.sp, (adr+2)); \
z80.reg.pc = z80_word; \
}

#define CALL_Z(adr) { \
if((z80.reg.f & FLAG_Z) == FLAG_Z) { \
	z80_word = RD_WORD_MEM(adr); \
	z80.reg.sp-=2; \
	WR_WORD_MEM(z80.reg.sp, (adr+2)); \
	z80.reg.pc = z80_word; \
} else { \
	z80.reg.pc+=3; \
} \
}

#define CALL_NZ(adr) { \
if((z80.reg.f & FLAG_Z) == 0) { \
	z80_word = RD_WORD_MEM(adr); \
	z80.reg.sp-=2; \
	WR_WORD_MEM(z80.reg.sp, (adr+2)); \
	z80.reg.pc = z80_word; \
} else { \
	z80.reg.pc+=3; \
} \
}

#define CALL_C(adr) { \
if((z80.reg.f & FLAG_C) == FLAG_C) { \
	z80_word = RD_WORD_MEM(adr); \
	z80.reg.sp-=2; \
	WR_WORD_MEM(z80.reg.sp, (adr+2)); \
	z80.reg.pc = z80_word; \
} else { \
	z80.reg.pc+=3; \
} \
}

#define CALL_NC(adr) { \
if((z80.reg.f & FLAG_C) == 0) { \
	z80_word = RD_WORD_MEM(adr); \
	z80.reg.sp-=2; \
	WR_WORD_MEM(z80.reg.sp, (adr+2)); \
	z80.reg.pc = z80_word; \
} else { \
	z80.reg.pc+=3; \
} \
}


//------------------------------------------------
// RET
//------------------------------------------------

#define RET(adr) { \
z80.reg.pc=RD_WORD_MEM(adr); \
z80.reg.sp+=2; \
z80.halt_skip = 1; \
}


#define RET_Z(adr) { \
if((z80.reg.f & FLAG_Z) == FLAG_Z) { \
	z80.reg.pc=RD_WORD_MEM(adr); \
	z80.reg.sp+=2; \
	z80.halt_skip = 1; \
} else { \
	z80.reg.pc++; \
} \
}

#define RET_NZ(adr) { \
if((z80.reg.f & FLAG_Z) == 0) { \
	z80.reg.pc=RD_WORD_MEM(adr); \
	z80.reg.sp+=2; \
	z80.halt_skip = 1; \
} else { \
	z80.reg.pc++; \
} \
}

#define RET_C(adr) { \
if((z80.reg.f & FLAG_C) == FLAG_C) { \
	z80.reg.pc=RD_WORD_MEM(adr); \
	z80.reg.sp+=2; \
	z80.halt_skip = 1; \
} else { \
	z80.reg.pc++; \
} \
}

#define RET_NC(adr) { \
if((z80.reg.f & FLAG_C) == 0) { \
	z80.reg.pc=RD_WORD_MEM(adr); \
	z80.reg.sp+=2; \
	z80.halt_skip = 1; \
} else { \
	z80.reg.pc++; \
} \
}

//------------------------------------------------
// RETI
//------------------------------------------------

#define RETI(adr) { \
z80.ime_flag = 1; \
z80.reg.pc=RD_WORD_MEM(adr); \
z80.reg.sp+=2; \
z80.halt_skip = 1; \
}

//------------------------------------------------
// DAA
//------------------------------------------------

#define DAA { \
z80_word = z80.reg.a; \
z80_word |= ((z80.reg.f & FLAG_C) ? DAA_TABLE_FLAG_C : 0); \
z80_word |= ((z80.reg.f & FLAG_H) ? DAA_TABLE_FLAG_H : 0); \
z80_word |= ((z80.reg.f & FLAG_N) ? DAA_TABLE_FLAG_N : 0); \
z80.reg.af = daa_flag_cnh[z80_word]; \
z80.reg.pc++; \
}


//------------------------------------------------
// DI, EI
//------------------------------------------------

#define DI {z80.ime_flag = 0; z80.reg.pc++;}

#define EI {z80.ime_flag = 1; z80.reg.pc++;}


//------------------------------------------------
// SCF, CCF
//------------------------------------------------

#define SCF {z80.reg.f = (z80.reg.f & FLAG_Z) | FLAG_C; z80.reg.pc++;}

#define CCF {z80.reg.f = (z80.reg.f & FLAG_Z) | ((z80.reg.f & FLAG_C) ? 0 : FLAG_C); z80.reg.pc++;}




//------------------------------------------------
// POP, PUSH
//------------------------------------------------

#define POP_DR(adr) {adr=RD_WORD_MEM(z80.reg.sp); z80.reg.sp+=2; z80.reg.pc++;}

#define PUSH_DR(adr) { \
z80.reg.sp-=2; \
WR_WORD_MEM(z80.reg.sp, adr); \
z80.reg.pc++; \
}


//------------------------------------------------
// RST
//------------------------------------------------

#define RST(adr,value) { \
z80.reg.sp-=2; \
WR_WORD_MEM(z80.reg.sp, adr); \
z80.reg.pc=value; \
}


//------------------------------------------------
// mcu cycles for each opcode
//------------------------------------------------
static const uint8_t cycles[256] = {
	4,12,8,8,4,4,8,4,20,8,8,8,4,4,8,4,
	4,12,8,8,4,4,8,4,12,8,8,8,4,4,8,4,
	12,12,8,8,4,4,8,4,12,8,8,8,4,4,8,4,
	12,12,8,8,12,12,12,4,12,8,8,8,4,4,8,4,
	4,4,4,4,4,4,8,4,4,4,4,4,4,4,8,4,
	4,4,4,4,4,4,8,4,4,4,4,4,4,4,8,4,
	4,4,4,4,4,4,8,4,4,4,4,4,4,4,8,4,
	8,8,8,8,8,8,4,8,4,4,4,4,4,4,8,4,
	4,4,4,4,4,4,8,4,4,4,4,4,4,4,8,4,
	4,4,4,4,4,4,8,4,4,4,4,4,4,4,8,4,
	4,4,4,4,4,4,8,4,4,4,4,4,4,4,8,4,
	4,4,4,4,4,4,8,4,4,4,4,4,4,4,8,4,
	20,12,16,16,24,16,8,16,20,16,16,4,24,24,8,16,
	20,12,16,1,24,16,8,16,20,16,16,1,24,1,8,16,
	12,12,8,1,1,16,8,16,16,4,16,1,1,1,8,16,
	12,12,8,4,1,16,8,16,12,8,16,4,1,1,8,16
};


#endif /* Z80_OPCODE_H_ */
